/*
 * SAProf.CPP
 * Autor:               Stefan Milcke
 * Erstellt am:         14.03.2000
 * Letzte Aenderung am: 11.04.2000
 *
*/

/* defines for OS/2 include */

/* Standard headers */
#include <sys\stat.h>

/* UI Class Library headers */
#include <itrace.hpp>
#include <iexcbase.hpp>
#include <fstream.h>

/* Application headers */
#include "SAProf.hpp"

/* OS/2 Headers */

static bool gTimeCheckingEnabled=true;

/******************************************************************************/
/* Class SAProfileEntry                                                       */
/******************************************************************************/
class SAProfileEntry
{
 public:
  friend class SAProfileSection;
  friend class SAProfile;
  friend class SAProfile::Cursor;
  SAProfileEntry(const char *entryLine);
  virtual ~SAProfileEntry();
  virtual IString name()
  { return iKey; }
  virtual SAProfileEntry &setValue(const IString &aValue)
  { iValue=aValue; return *this; }
  virtual IString value()
  { return iValue; }
 protected:
 private:
  IString iKey;
  IString iValue;
  bool iNoKey;
};

/******************************************************************************/
/* Class SAProfileSection                                                     */
/******************************************************************************/
class SAProfileSection
{
 public:
  friend class SAProfile;
  friend class SAProfile::Cursor;
  SAProfileSection(const char *section);
  virtual ~SAProfileSection();
  virtual IString name() const
  { return iSection; }
  virtual long numberOfKeys() const
  { return iEntryList.numberOfElements(); }
  virtual bool containsKeyName(const char *key) const
  { if(findKey(key)) return true; else return false; }
  virtual IString elementWithKey(const char *key);
  virtual SAProfileSection &addOrReplaceElementWithKey(const char *key
                                                       ,const IString &data);
 protected:
  virtual long findKey(const char *key) const;
 private:
  IString iSection;
  SAProfileEntryList iEntryList;
};

/******************************************************************************/
/* Implementation of class SAProfile                                          */
/******************************************************************************/
//------------------------------- constructor 1 --------------------------------
SAProfile::SAProfile(const SAProfile &aProfile)
 : iName(aProfile.name()),iChanged(aProfile.iChanged),iTS(0)
  ,iResolveVariableReferences(aProfile.isResolveVariableReferencesEnabled())
{
 IFUNCTRACE_DEVELOP();
 reReadProfile();
}

//------------------------------- constructor 2 --------------------------------
SAProfile::SAProfile(const char *profileName,bool resolveVariableSubstitutions)
 : iName(profileName),iChanged(false),iTS(0)
  ,iResolveVariableReferences(resolveVariableSubstitutions)
{
 IFUNCTRACE_DEVELOP();
 reReadProfile();
}

//----------------------------- applicationOrKeyAt -----------------------------
IString SAProfile::applicationOrKeyAt(const SAProfile::Cursor &cursor) const
{
 IFUNCTRACE_DEVELOP();
 if(cursor.isValid())
 {
  if(cursor.iApplName.length())
   return iSectionList.elementAtPosition(cursor.iSectionPos)
            ->iEntryList.elementAtPosition(cursor.iEntryPos)->name();
  else
   return iSectionList.elementAtPosition(cursor.iSectionPos)->name();
 }
 else
  return IString("");
}

//--------------------------------- destructor ---------------------------------
SAProfile::~SAProfile()
{
 IFUNCTRACE_DEVELOP();
 purge();
}

//-------------------------------- numberOfKeys --------------------------------
unsigned long SAProfile::numberOfKeys(const char *applName) const
{
 IFUNCTRACE_DEVELOP();
 long i=findApplication(applName);
 if(i)
  return iSectionList.elementAtPosition(i)->numberOfKeys();
 else
  return 0;
}

//------------------------- addOrReplaceElementWithKey -------------------------
SAProfile &SAProfile::addOrReplaceElementWithKey(const char *key
                                                 ,const IString &data
                                                 ,const char *applName)
{
 IFUNCTRACE_DEVELOP();
 IString a(appNameFromArg(applName));
 long i=findApplication(a);
 if(!i)
 {
  iSectionList.addAsLast(new SAProfileSection("["+a+"]"));
  i=findApplication(a,false);
 }
 iSectionList.elementAtPosition(i)->addOrReplaceElementWithKey(key,data);
 writeProfile();
 return *this;
}

//------------------------------ containsKeyName -------------------------------
bool SAProfile::containsKeyName(const char *key,const char *applName) const
{
 IFUNCTRACE_DEVELOP();
 long i=findApplication(applName);
 if(i)
  return iSectionList.elementAtPosition(i)->containsKeyName(key);
 else
  return false;
}

//---------------------------- containsApplication -----------------------------
bool SAProfile::containsApplication(const char *applName) const
{
 IFUNCTRACE_DEVELOP();
 if(findApplication())
  return true;
 else
  return false;
}

//------------------------ deleteElementWithApplication ------------------------
SAProfile &SAProfile::deleteElementWithApplication(const char *applName)
{
 IFUNCTRACE_DEVELOP();
 long i=findApplication(applName);
 if(i)
 {
  delete iSectionList.elementAtPosition(i);
  iSectionList.removeAtPosition(i);
  writeProfile();
 }
 return *this;
}

//---------------------------- deleteElementWithKey ----------------------------
SAProfile &SAProfile::deleteElementWithKey(const char *key,const char *applName)
{
 IFUNCTRACE_DEVELOP();
 long i=findApplication(applName);
 if(i)
 {
  long j=iSectionList.elementAtPosition(i)->findKey(key);
  if(j)
  {
   delete iSectionList.elementAtPosition(i)->iEntryList.elementAtPosition(j);
   iSectionList.elementAtPosition(i)->iEntryList.removeAtPosition(j);
   writeProfile();
  }
 }
 return *this;
}

//------------------------------- elementWithKey -------------------------------
IString SAProfile::elementWithKey(const char *key,const char *applName)
{
 IFUNCTRACE_DEVELOP();
 long i=findApplication(applName);
 if(i)
 {
  IString el(iSectionList.elementAtPosition(i)->elementWithKey(key));
  if(iResolveVariableReferences)
  {
   unsigned long s,e;
   s=el.indexOf("%[");
   e=el.indexOf("]%");
   if(s && e)
   {
    IString tmp(el.subString(s+2,e-s-2));
    IString a(tmp.subString(1,tmp.indexOf(".")-1));
    if(a.length()==0)
     a=appNameFromArg(applName);
    IString k(tmp.subString(tmp.indexOf(".")+1));
    IString v=elementWithKey(k,a);
    el.change(IString("%["+tmp+"]%"),v);
   }
  }
  return el;
 }
 else
  return IString("");
}

//------------------------------- integerWithKey -------------------------------
long SAProfile::integerWithKey(const char *key,const char *applName)
{
 IFUNCTRACE_DEVELOP();
 return elementWithKey(key,applName).asInt();
}

//---------------------- enableResolveVariableReferences -----------------------
SAProfile &SAProfile::enableResolveVariableReferences(bool enable)
{
 IFUNCTRACE_DEVELOP();
 iResolveVariableReferences=enable;
 return *this;
}

//--------------------------- isTimeCheckingEnabled ----------------------------
bool SAProfile::isTimeCheckingEnabled()
{
 IFUNCTRACE_DEVELOP();
 return gTimeCheckingEnabled;
}

//----------------------------- enableTimeChecking -----------------------------
void SAProfile::enableTimeChecking(bool enable)
{
 IFUNCTRACE_DEVELOP();
 gTimeCheckingEnabled=enable;
}

//------------------------------- reReadProfile --------------------------------
void SAProfile::reReadProfile() const
{
 IFUNCTRACE_DEVELOP();
 double ts;
 {
  struct stat buf;
  FILE *fp;
  int fh,result;
  fp=fopen(name(),"r");
  fstat(fileno(fp),&buf);
  fclose(fp);
  ts=buf.st_mtime;
 }
 if(ts!=iTS)
 {
  purge();
  ifstream fs;
  IString tmp;
  bool bHeading=true;
  fs.open(name());
  while(!fs.eof() && !fs.fail() && bHeading)
  {
   tmp=IString::lineFrom(fs);
   if(tmp.subString(1,1)=="[" && tmp.subString(tmp.length(),1)=="]")
    bHeading=false;
   else
    iHeadingList.addAsLast(new IString(tmp));
  }
  while(!fs.eof() && !fs.fail())
  {
   if(tmp.subString(1,1)=="[" && tmp.subString(tmp.length(),1)=="]")
   {
    iSectionList.addAsLast(new SAProfileSection(tmp));
    long p=iSectionList.numberOfElements();
    SAProfileEntryList *list=&(iSectionList.elementAtPosition(p)->iEntryList);
    tmp=IString::lineFrom(fs);
    while(!fs.eof() && !fs.fail()
          &&!(tmp.subString(1,1)=="[" && tmp.subString(tmp.length(),1)=="]"))
    {
     if(tmp.length())
      list->addAsLast(new SAProfileEntry(tmp));
     tmp=IString::lineFrom(fs);
    }
   }
   else
    tmp=IString::lineFrom(fs);
  }
  fs.close();
  iTS=ts;
 }
}

//-------------------------------- writeProfile --------------------------------
void SAProfile::writeProfile()
{
 IFUNCTRACE_DEVELOP();
 long i,j;
 ofstream fs;
 fs.open(name());

 // write headers
 for(i=1
     ;i<=iHeadingList.numberOfElements() && !fs.fail()
     ;i++)
 {
  fs << (*(iHeadingList.elementAtPosition(i))) << "\n";
 }
 // write sections
 for(i=1
     ;i<=iSectionList.numberOfElements() && !fs.fail()
     ;i++)
 {
  fs << IString("["+iSectionList.elementAtPosition(i)->name()+"]") << "\n";
  // write entries
  SAProfileEntryList *list=&(iSectionList.elementAtPosition(i)->iEntryList);
  for(j=1
      ;j<=list->numberOfElements() && !fs.fail()
      ;j++)
  {
   IString line(list->elementAtPosition(j)->name()
                +"="
                +list->elementAtPosition(j)->value());
   if(line!="=")
    fs << line << "\n";
  }
  fs << "\n";
 }
 fs.close();
 {
  struct stat buf;
  FILE *fp;
  int fh,result;
  fp=fopen(name(),"r");
  fstat(fileno(fp),&buf);
  fclose(fp);
  iTS=buf.st_mtime;
 }
 _heapmin();
}

//------------------------------ findApplication -------------------------------
long SAProfile::findApplication(const char *applName,bool reRead) const
{
 IFUNCTRACE_DEVELOP();
 if(reRead)
  if(isTimeCheckingEnabled())
   reReadProfile();
 long i=0;
 IString a(appNameFromArg(applName));
 for(i=1;i<=iSectionList.numberOfElements();i++)
  if(iSectionList.elementAtPosition(i)->name()==a)
   return i;
 return 0;
}

//----------------------------------- purge ------------------------------------
void SAProfile::purge() const
{
 IFUNCTRACE_DEVELOP();
 while(iHeadingList.numberOfElements())
 {
  delete iHeadingList.elementAtPosition(1);
  iHeadingList.removeAtPosition(1);
 }
 while(iSectionList.numberOfElements())
 {
  delete iSectionList.elementAtPosition(1);
  iSectionList.removeAtPosition(1);
 }
}

/******************************************************************************/
/* Implementation of class SAProfileSection                                   */
/******************************************************************************/
//-------------------------------- constructor ---------------------------------
SAProfileSection::SAProfileSection(const char *section)
{
 IFUNCTRACE_DEVELOP();
 iSection=IString(section).subString(2,IString(section).length()-2);
}

//--------------------------------- destructor ---------------------------------
SAProfileSection::~SAProfileSection()
{
 IFUNCTRACE_DEVELOP();
 while(iEntryList.numberOfElements())
 {
  delete iEntryList.elementAtPosition(1);
  iEntryList.removeAtPosition(1);
 }
}

//------------------------------- elementWithKey -------------------------------
IString SAProfileSection::elementWithKey(const char *key)
{
 IFUNCTRACE_DEVELOP();
 long i=findKey(key);
 if(i)
  return iEntryList.elementAtPosition(i)->value();
 else
  return IString("");
}

//------------------------- addOrReplaceElementWithKey -------------------------
SAProfileSection &SAProfileSection::addOrReplaceElementWithKey(const char *key
                                                               ,const IString &data)
{
 IFUNCTRACE_DEVELOP();
 long i=findKey(key);
 if(!i)
  iEntryList.addAsLast(new SAProfileEntry(IString(key)+"="+data));
 else
  iEntryList.elementAtPosition(i)->setValue(data);
 return *this;
}

//---------------------------------- findKey -----------------------------------
long SAProfileSection::findKey(const char *key) const
{
 IFUNCTRACE_DEVELOP();
 long i;
 for(i=1;i<=iEntryList.numberOfElements();i++)
  if(iEntryList.elementAtPosition(i)->name()==key)
   return i;
 return 0;
}

/******************************************************************************/
/* Implementation of class SAProfileEntry                                     */
/******************************************************************************/
//-------------------------------- constructor ---------------------------------
SAProfileEntry::SAProfileEntry(const char *entryLine)
 : iNoKey(false)
{
 IFUNCTRACE_DEVELOP();
 IString tmp(entryLine);
 if(tmp.length() && tmp.indexOf("="))
 {
  iKey=tmp.subString(1,tmp.indexOf("=")-1);
  iValue=tmp.subString(iKey.length()+2);
 }
 else
 {
  iKey=tmp;
  iValue="";
  if(!tmp.indexOf("="))
   iNoKey=true;
 }
}

//--------------------------------- destructor ---------------------------------
SAProfileEntry::~SAProfileEntry()
{
 IFUNCTRACE_DEVELOP();
}

/******************************************************************************/
/* Implementation of class SAProfile::Cursor                                  */
/******************************************************************************/
//------------------------------- constructor 1 --------------------------------
SAProfile::Cursor::Cursor(SAProfile &profile)
 : iTS(profile.iTS),iProfile(&profile),iApplName("")
{
 IFUNCTRACE_DEVELOP();
}

//------------------------------- constructor 2 --------------------------------
SAProfile::Cursor::Cursor(SAProfile &profile,const char *applName)
 : iTS(profile.iTS),iProfile(&profile),iApplName(applName)
{
 IFUNCTRACE_DEVELOP();
}

//--------------------------------- destructor ---------------------------------
SAProfile::Cursor::~Cursor()
{
 IFUNCTRACE_DEVELOP();
}

//--------------------------------- setToFirst ---------------------------------
bool SAProfile::Cursor::setToFirst()
{
 IFUNCTRACE_DEVELOP();
 iTS=0;
 if(iApplName.length())
 {
  iSectionPos=iProfile->findApplication(iApplName);
  if(iSectionPos &&
     iProfile->iSectionList.elementAtPosition(iSectionPos)->iEntryList.numberOfElements())
  {
   iEntryPos=1;
   iTS=iProfile->iTS;
   return true;
  }
  else
   return false;
 }
 else
 {
  if(iProfile->iSectionList.numberOfElements())
  {
   iSectionPos=1;
   iEntryPos=1;
   iTS=iProfile->iTS;
   return true;
  }
  else
   return false;
 }
}

//--------------------------------- setToNext ----------------------------------
bool SAProfile::Cursor::setToNext()
{
 IFUNCTRACE_DEVELOP();
 iTS=0;
 if(iApplName.length())
 {
  iSectionPos=iProfile->findApplication(iApplName);
  if(iSectionPos &&
     iProfile->iSectionList.elementAtPosition(iSectionPos)->iEntryList.numberOfElements()>iEntryPos)
  {
   iEntryPos++;
   iTS=iProfile->iTS;
   return true;
  }
  else
   return false;
 }
 else
 {
  if(iProfile->iSectionList.numberOfElements()>iSectionPos)
  {
   iSectionPos++;
   iTS=iProfile->iTS;
   return true;
  }
  else
   return false;
 }
 return true;
}

//------------------------------- setToPrevious --------------------------------
bool SAProfile::Cursor::setToPrevious()
{
 IFUNCTRACE_DEVELOP();
 iTS=0;
 if(iApplName.length())
 {
  if(iEntryPos)
  {
   iEntryPos--;
   iTS=iProfile->iTS;
   return true;
  }
  else
   return false;
 }
 else
 {
  if(iSectionPos>1)
  {
   iSectionPos--;
   iTS=iProfile->iTS;
   return true;
  }
  else
   return false;
 }
}

//--------------------------------- setToLast ----------------------------------
bool SAProfile::Cursor::setToLast()
{
 IFUNCTRACE_DEVELOP();
 iTS=0;
 if(iApplName.length())
 {
  iSectionPos=iProfile->findApplication(iApplName);
  if(iSectionPos)
  {
   iEntryPos=iProfile->iSectionList.elementAtPosition(iSectionPos)->iEntryList.numberOfElements();
   if(iEntryPos)
   {
    iTS=iProfile->iTS;
    return true;
   }
   else
    return false;
  }
  else
   return false;
 }
 else
 {
  iSectionPos=iProfile->iSectionList.numberOfElements();
  if(iSectionPos)
  {
   iTS=iProfile->iTS;
   return true;
  }
  else
   return false;
 }
}

//---------------------------------- isValid -----------------------------------
bool SAProfile::Cursor::isValid() const
{
 IFUNCTRACE_DEVELOP();
 if(!iTS || iProfile->iTS!=iTS)
  return false;
 else
  return true;
}

//--------------------------------- invalidate ---------------------------------
void SAProfile::Cursor::invalidate()
{
 IFUNCTRACE_DEVELOP();
 iTS=0;
}
